var csInterface = new CSInterface();

// Nodejs >= 7.7.4 requires path to be included as absolute path. Determining the absolute location of js files.
var loc = window.location.pathname;
var OSVersion = csInterface.getOSInformation();
var dir;
if (OSVersion.indexOf("Windows") >= 0)
{
  dir = decodeURI(loc.substring(1, loc.lastIndexOf('/')));
}
else if (OSVersion.indexOf("Mac") >= 0)
{
  dir = decodeURI(loc.substring(0, loc.lastIndexOf('/')));
}


var NetworkHttpService = require(dir + '/upload/NetworkHttpService');
var utils = require(dir + '/upload/utils');
var UploadError = require(dir + '/upload/UploadError');
var agent = require(dir + '/upload/ProxyAgent');
var kerberos = require(dir + '/upload/Kerberos');
var Q = require('q');
var https = require('https');
var Path = require('path');
var fs = require('fs');
var AWS = require('aws-sdk');
var crypt = require('crypto');
var qlimit = require('qlimit')(10); // limit AWS upload at 10 simultaneous connections at a time

// CONSTANTS
var hostname = 'https://cc-api-cp.adobe.io';
var S3_CHUNK_SIZE = 5 * 1024 * 1024; // 5 MB
var COMMUNITY_ID = 'lastmile';
var api_key = 'xxx';
var CHUNK_UPLOAD_MAX_RETRY = 2;
var PUBLISH_RESOURCE_TYPE = 'application/vnd.adobe.web-package';
https.globalAgent.maxSockets = 20; // AWS upload at 10 simultaneous, CP upload at 5 simultaneous plus 5 buffer

// URLs
var GET_FILE_SIZE_URL = '/api/v2/limits';
var UPLOAD_FILE_URL = '/api/v2/resources/';
var UPLOAD_S3_URL_SUFFIX = '/:S3';
var PUBLISH_CP_URL = '/api/v2/' + COMMUNITY_ID + '/assets/';
var webDocHost = 'https://indd.adobe.com';
var coverImagePath = '/Thumbnail/Cover/cover.png';
var MANIFEST_FILE = 'manifest.json';
var FINALIZE_ENDPOINT = "/api/v1/idpubs/finalizePublish";
var LOGDATA_ENDPOINT = "/api/v1/idpubs/logData";


var STAGE_IMS_SERVER =		"https://ims-na1-stg1.adobelogin.com/";
var PROD_IMS_SERVER	= 		"https://adobeid-na1.services.adobe.com/";
var IMS_PROFILE_END_POINT = "ims/profile/v1?";

// Error handling
var kSuccess = "Success", kFailure = "Failure", kCancel = "Cancel";
var kUnknownError = "Unknown Error";
var kAuthenticationError = "Authentication Error";
var kNetworkError = "Network Error";

// Don't change the numbers. Just append in the end
var kCatchResolvePathErr = 1; 					//"Catch block for realpathSync call";
var kLastUploadPendingErr = 2; 					//"Last upload is still pending";
var kParsingFilesListErr = 3; 					//"Error in parsing list of files to upload.";
var kPublishingPackageErr = 4; 					//"Error in publishing package";
var kCatchPublishingPackageErr = 5;  			//"Catch in publishing package";
var kUnextectedEventDataErr = 6;				//"Unexpected data in event";
var kInvalidResponseCPPublishErr = 7;			//"Invalid Response for CP publish API";
var kMissingAssetIDErr = 8; 					//"Asset id missing";
var kCatchPublishingPackageErr2 = 9; 			//"Catch in publishing package 2"
var kFindFileSizeLimitErr = 10;					//"findFileSizeLimit failed"
var kInvalidResponseFileSizeLimitErr = 11;		//"Invalid response in querying the file limit"
var kFileReadErr = 12; 							//"Error in reading file"
var kInvalidFilePathErr = 13; 					//"The files to upload has a path which is not a file"
var kParsingS3DetailsFromCPErr = 14; 			//"Error in Parsing S3 details from CP"
var kReadingUploadingInChunkErr = 15; 			//"Error in reading heavy file and uploading in chunk"
var kFinalizingUploadS3WithStatusCodeErr = 16; 	//"Finalizing upload to S3 failed with status code"
var kFinalizingUploadS3Err = 17; 				//"Error in finalizing upload to S3"
var kPublishedWithNoProperResponseErr = 18; 	//"Published successfully, but with no proper response"
var kTaskFailedErr = 19; 						//"Task failed"
var kCatchMimeTypeCalErr = 20; 					//"Catch block for getResolvedFilePath call"
var kFindFileSizeLimitErr2 = 21;				//"Error in getting FindFileSizeLimit
var kCatchFindFileSizeLimitErr = 22;			//"Catch block for FindFileSizeLimit"
var kCatchUploadFileViaCPErr = 23; 				//"Promise catch in upload file via CP"
var kCatchUploadFileDirectlyToS3Err = 24;		//"Promise catch in upload file directly to S3"
var kPublishTOCPCatchErr = 25;					//"Published promise catch"
var kCatchCPPublishErr = 26;					//"CP publish Catch"
var kMismatchInFileAndPackagePathErr = 27;		//"Mismatch in file and package path error"
var kCatchOnPublishHTMLErr = 28;				//"Catch in publishing
var kInvalidResponseMetadataErr = 29;			//"Invalid response in fetching the metadata"
var kCatchFetchMetadataErr = 30;				//"Catch block for fetchAssetMetadata"
var kFetchMetadataErr = 31;						//"fetchAssetMetadata failed"
var kRePublishArtworkCatchErr = 32;				//"Re-Publishing update artwork promise catch"
var kRePublishArtworkResponseErr = 33;			//"Re-Publishing update artwork Invalid response"
var kRePublishMetadataCatchErr = 34;			//"Re-Publishing update metadata promise catch"
var kRePublishMetadataResponseErr = 35;			//"Re-Publishing update metadata Invalid response"
var kManifestFileNotExistsErr = 36;				//"Manifest file doesn't exists at the expected path"
var kParsingMachineTagsListErr = 37;            // Error in parsing machine tags sent from plugin
var kTokenVerificationFailure = 38;				// Token authentication failed with various error responses
var kTokenVerificationNetworkFailure = 39; 		// Token authentication failed with network issues like connection reset, timeout etc
var kUploadChunkS3Err = 40;						// Upload to S3 of a chunk failed for a big file				

var kMaxNumberOfRetriesForUpload = 3;

// Globals
var currentPublish;
var publishOnlineLogObject = {
	"creatorName"           : "",
	"idVersion"        		: "",
	"resourcePath"          : "",
	"asset_id"              : "",
	"finalStatus"           : "",
	"statusDescription"     : "",
	"isRepublishing"        : false,
	"httpRequestLogData"    : []
};
var extToMimes = {
	'jpeg': 'image/jpeg',
	'jpg': 'image/jpeg',

	'html' : 'text/html',
	'htm' : 'text/html',
	'shtml' : 'text/html',
	'css': 'text/css',
	'xml' : 'text/xml',

	'gif' : 'image/gif',
	'js' : 'application/javascript',
	'atom' : 'application/atom+xml',
	'rss': 'application/rss+xml',

	'mml' : 'text/mathml',
	'txt' : 'text/plain',
	'jad' : 'text/vnd.sun.j2me.app-descriptor',
	'wml': 'text/vnd.wap.wml',
	'htc' : 'text/x-component',
	'appcache' : 'text/cache-manifest',

	'png': 'image/png',
	'tif': 'image/tiff',
	'tiff': 'image/tiff',
	'wbmp': 'image/vnd.wap.wbmp',
	'ico': 'image/x-icon',
	'jng': 'image/x-jng',
	'bmp': 'image/x-ms-bmp',
	'svg': 'image/svg+xml',
	'svgz': 'image/svg+xml',
	'webp': 'image/webp',


	'woff': 'application/font-woff',
	'json': 'application/json',
	'doc': 'application/msword',
	'pdf': 'application/pdf',
	'ps': 'application/postscript',
	'eps': 'application/postscript',
	'ai': 'application/postscript',
	'rtf': 'application/rtf',
	'm3u8': 'application/vnd.apple.mpegurl',
	'xls': ' application/vnd.ms-excel',
	'eot': 'application/vnd.ms-fontobject',
	'ppt': 'application/vnd.ms-powerpoint',
	'wmlc': 'application/vnd.wap.wmlc',
	'swf': 'application/x-shockwave-flash',
	'sit': 'application/x-stuffit',
	'xhtml': 'application/xhtml+xml',
	'xspf': 'application/xspf+xml',
	'zip': 'application/zip',

	'docx': 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
	'xlsx': 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
	'pptx': 'application/vnd.openxmlformats-officedocument.presentationml.presentation',


	'mid': 'audio/midi',
	'midi': 'audio/midi',
	'kar': 'audio/midi',
	'mp3': 'audio/mpeg',
	'ogg': 'audio/ogg',
	'm4a': 'audio/x-m4a',
	'ra': 'audio/x-realaudio',

	'3gpp': 'video/3gpp',
	'3gp': 'video/3gpp',
	'ts': 'video/mp2t',
	'mp4': 'video/mp4',
	'mpeg': 'video/mpeg',
	'mpg': 'video/mpeg',
	'mov': 'video/quicktime',
	'webm': 'video/webm',
	'flv': 'video/x-flv',
	'm4v': 'video/x-m4v',
	'mng': 'video/x-mng',
	'asx': 'video/x-ms-asf',
	'asf': 'video/x-ms-asf',
	'wmv': 'video/x-ms-wmv',
	'avi': 'video/x-msvideo'
};

var httpService;

function throwError(baseError, code, errInput)
{
	var detailedMessage = getDetailedErrorString(baseError, code, errInput);
	var err;
	if (errInput)
	{
		err = errInput;
		err.message = detailedMessage;
	}
	else
		err = new Error(detailedMessage);
	err.id_generated = true;
	throw err;
}
function onLoaded() {
	var appName = csInterface.hostEnvironment.appName;

	if (appName !== "FLPR") {
		loadJSX();
	}

	csInterface.addEventListener("com.adobe.events.PublishHTML.indesign", evtHandler);

	httpService = new NetworkHttpService(authCallback);
	httpService.setTimeout(300000);       // 300 Seconds

	sendExtenionLoadedEvent();
}

/**
 * Load JSX file into the scripting context of the product. All the jsx files in
 * folder [ExtensionRoot]/jsx will be loaded.
 */
function loadJSX() {
	var extensionRoot = csInterface.getSystemPath(SystemPath.EXTENSION) + "/jsx/";
	csInterface.evalScript('$._ext.evalFiles("' + extensionRoot + '")');
}

function evalScript(script, callback) {
	new CSInterface().evalScript(script, callback);
}

function sendEvent(data)
{
	var event = new CSEvent("com.adobe.events.HTMLPublish.html", "APPLICATION");
	event.extensionId = csInterface.getExtensionID();

	event.data = JSON.stringify(data);
	csInterface.dispatchEvent(event);
}

function sendExtenionLoadedEvent()
{
	var extLoadedEvt = {};
	extLoadedEvt.type = "Extension Loaded";

	sendEvent(extLoadedEvt);
}

function sendProgressEvent(percent)
{
	var progInfo = {};
	progInfo.percent = percent;
	progInfo.type = "Upload Progress";

	sendEvent(progInfo);
}
function sendPublishedDocURL(assetid)
{
	var pubDocUrl = {};
	pubDocUrl.docUrl = webDocHost + "/view/" + assetid;
	pubDocUrl.type = "Published Doc URL";

	sendEvent(pubDocUrl);
}
function sendPublishPkgSize(pkgSize) {
	var packageSizeInfo = {};
	packageSizeInfo.pkgSize = pkgSize;
	packageSizeInfo.type = "Published Package Size";

	sendEvent(packageSizeInfo);
}

function sendCompletionEvent(status, data) {
	var complEvt = {};
	complEvt.type = "Uploading Complete";
	complEvt.status = status;
	var finalEvt = $.extend(complEvt, data);
	sendEvent(finalEvt);
}

function sendPublishCompletionEvent(status, data)
{
	if(status == kCancel) {
		sendCompletionEvent(status, data);
		return;
	}

	publishOnlineLogObject.finalStatus = status;
	if(status == kFailure) {
		publishOnlineLogObject.statusDescription = data.detailedReason;
	}

	data.resourcePath = publishOnlineLogObject.resourcePath ? publishOnlineLogObject.resourcePath : '';

	var logBody = JSON.stringify(publishOnlineLogObject);
	console.log("final global obj = " + logBody);

	invokeHttpRequest('POST', webDocHost + LOGDATA_ENDPOINT, {
		'content-type': 'application/json',
		'content-length': logBody.length
	}, logBody, { responseType: 'json' }, false, true)
			.then(function () {
				sendCompletionEvent(status, data);
			})
			.catch(function (err) {
				console.log("Error while trying to call logData API: " + err);
				sendCompletionEvent(status, data);
			});
}

var kerberosPromiseDict = {};
function sendKerberosTokenRequest(deferred, proxyServer)
{
	var uuid = utils.generateUuid();
	kerberosPromiseDict[uuid] = deferred;
	var data = {};
	data.type = "Kerberos Token Request";
	data.uuid = uuid;
	data.proxyServer = proxyServer;
	sendEvent(data);
}

function handleKerberosTokenResponse(evtdata)
{
	if (evtdata.status == "success") {
		kerberosPromiseDict[evtdata.uuid].resolve(evtdata.token);
	}
	else {
		publishOnlineLogObject.httpRequestLogData.push(
				{
					error: "Kerberos_Token_Generation_Error",
					errorMessage: evtdata.errorMessage
				}
		);
		kerberosPromiseDict[evtdata.uuid].reject(evtdata.errorMessage);
	}
	delete kerberosPromiseDict[evtdata.uuid];
}

function cancelUpload() {
	if (currentPublish) {
		currentPublish.abort();
		currentPublish = undefined;
	}
}

function evtHandler(event)
{
	var evtData = event.data;
	if (evtData == "Cancel Publish")
	{
		if (currentPublish)
			cancelUpload();
		else
			console.log('Cancel event from PP when upload is not in progress');
		sendPublishCompletionEvent(kCancel, {});
	}
	else if (evtData == "InDesign Silently Publish Cancel")
	{
		if (currentPublish)
			cancelUpload();
		else
			console.log('InDesign silently Publish cancel event from PP when upload is not in progress');
		sendPublishCompletionEvent(kFailure, {reason: kUnknownError, detailedReason: "InDesign silently Publish cancel"});
	}
	else if (evtData.type == "Publish Document")
	{
		onPublishHTML(event);
	}
	else if (evtData.type == "Kerberos Token Response")
	{
		handleKerberosTokenResponse(evtData);
	}
}

function getDetailedErrorString(baseError, code, err) {

	var detailedError = "";
	if (err && err.id_generated && err.message)
	{
		detailedError += err.message;
	}
	else
	{
		detailedError += baseError;
		if (code)
		{
			detailedError += ' ErrorCode:' + code.code;
			if (code.message)
				detailedError += ' ' + code.message;
		}

		if (err)
		{
			detailedError += ' Error:';
			if (err.message)
				detailedError += err.message;
			else
				detailedError += err;
		}
	}

	console.log(detailedError);
	return detailedError;
}

function getMimeByExt(fileName) {
	try{
		var ext = Path.extname(fileName);
		ext = ext.substr(1);
		if (extToMimes.hasOwnProperty(ext)) {
			//console.log("Mime type Found. File Name: " + fileName + " " + extToMimes[ext]);
			return extToMimes[ext];
		}
		console.log("Unexpected extension in file name. File Name: " + fileName + " Ext: " + ext);
		return 'image/jpeg';
	} catch(err) {
		throwError('getMimeByExt:' + fileName, {code: kCatchMimeTypeCalErr}, err);
	}
}

function getResolvedFilePath(filepath) {

	try{
		return fs.realpathSync(filepath);
	} catch(err) {
		throwError('getResolvedFilePath:' + filepath, {code: kCatchResolvePathErr}, err);
	}
}

function onPublishHTML(event) {
	try{
		var detailedReason;
		if (currentPublish) {
			detailedReason = getDetailedErrorString('onPublishHTML:', {code: kLastUploadPendingErr});
			sendPublishCompletionEvent(kFailure, {reason: kUnknownError, detailedReason: detailedReason});
			return;
		}

		AWS.config.update({
			maxRetries: kMaxNumberOfRetriesForUpload
		});

		//alert("First alert for debug purpose");

		kerberos.SetupKerberosRequestHandler(sendKerberosTokenRequest);
		kerberos.SetupKerberosResponseHandler(handleKerberosTokenResponse);

		var publishInfo = event.data;
		if (publishInfo)
		{
			api_key = publishInfo.api_key;
			if (publishInfo.publishEnv !== "prod")
			{
				hostname = publishInfo.hostname;
				webDocHost = publishInfo.webDocHost;
			}
			currentPublish = new PublishPackage(publishInfo.token);
			coverImagePath = publishInfo.coverPath;

			if (publishInfo.proxyServer) {
				httpService.setProxyAgent(publishInfo.proxyServer, publishInfo.proxyUsername, publishInfo.proxyPassword); // Inform NetworkService
			}

			if (publishInfo.amazonProxyServer) {
				// Inform AWS SDK about proxies
				var amazonOptions = agent.CreateProxyAgent(publishInfo.amazonProxyServer,
																										publishInfo.amazonProxyServer === publishInfo.proxyServer ? publishInfo.proxyUsername : "",
																										publishInfo.amazonProxyServer === publishInfo.proxyServer ? publishInfo.proxyPassword : "");
				var httpOptionsDict = {};
				httpOptionsDict.agent = amazonOptions.agent;
				if (amazonOptions.ca && amazonOptions.ca.length > 0)
					httpOptionsDict.ca = amazonOptions.ca;
				AWS.config.update({
					httpOptions: httpOptionsDict
				});
			}
			else {
				// from: https://forums.aws.amazon.com/thread.jspa?messageID=334243
				// "Also, don't overuse a connection. Amazon S3 will accept up to 100 requests before it closes a connection (resulting in 'connection reset'). 
				// Rather than having this happen, use a connection for 80-90 requests before closing and re-opening a new connection."
				// 
				// If we do not set agent false, it will result in using the default agent that reuses connections. hence setting agent false so that no connection is reused.
				// This can have negative performance impact. However in our case, each of these connections will be used to upload 5MB of data, thus connection establishment time
				// is very small compared to overall operation. So we should be fine.
				AWS.config.update({
					httpOptions: {agent: false}
				});
			}

			// Comment from CEP team
			// Before CEP 6.1, we treat every attribute in event.data object as a regular string, but this is a bug opened by other customers.  
			// in CEP 6.1 we revised the behaviour that keep the type of each attribute of event.data as it was. 
			// In this case, the filesToUpload is a valid JSON string, so CEP parses it natively and turns it to a JavaScript object.
			var filesToUpload = publishInfo.filesToUpload;
			var machineTags = publishInfo.machineTags;

			// GET request to IMS servers, to authenticate the token
			var server = (publishInfo.publishEnv === "prod") ? PROD_IMS_SERVER : STAGE_IMS_SERVER;	
			var accessToken = "Bearer " + publishInfo.token;
			var httpHeader = {
				"Authorization": accessToken
			};		
			invokeHttpRequest('GET', server+IMS_PROFILE_END_POINT+"client_id="+api_key, httpHeader, undefined, { responseType: 'json' }, false)
			.then(function(response) {
				var message;
				if(response.statusCode >= 400 && response.statusCode <= 499) {
					message = response.statusCode.toString();
				}					
				else if(response.response) {
					var resp = JSON.parse(response.response);
					if(resp && resp.error) {
						message = resp.error;
					}
				}
				if(message) {
					var detailedReason = getDetailedErrorString('onPublishHTML:', {code: kTokenVerificationFailure, message: 'Error: '+ message});
					sendPublishCompletionEvent(kFailure, {reason: kAuthenticationError, detailedReason: detailedReason});
					currentPublish = undefined;
					return;
				}
				currentPublish.setGoogleAnalyticsParams(publishInfo.measurementId, publishInfo.enableCookieConsentBannerOption, publishInfo.cookieConsentBannerText);
				currentPublish.publish(publishInfo.packagePath, filesToUpload, publishInfo.title, publishInfo.description, machineTags,
					publishInfo.rePublishAssetID, publishInfo.idVersion, publishInfo.creatorName)
					.then(function(asset_id) {
						console.log('Asset ID: ' + asset_id);
						publishOnlineLogObject.asset_id = asset_id;
						sendPublishedDocURL(asset_id);
						sendPublishCompletionEvent(kSuccess, {});


					}, function(err) {
						console.log(err);
						if(err && (err.code === 'ENOTFOUND' || err.code === UploadError.NETWORK_ERROR)){
							sendPublishCompletionEvent(kFailure, {reason: kNetworkError});
						} else {
							detailedReason = getDetailedErrorString('onPublishHTML:', {code: kPublishingPackageErr, message: 'ErrorCode: ' + err ? err.code : ""}, err);
							sendPublishCompletionEvent(kFailure, {reason: kUnknownError, detailedReason: detailedReason});
						}
					})
					.catch(function(err) {
						// Unknonwn error from promise chain
						currentPublish.abort();
						detailedReason = getDetailedErrorString('onPublishHTML:', {code: kCatchPublishingPackageErr}, err);
						sendPublishCompletionEvent(kFailure, {reason: kUnknownError, detailedReason: detailedReason});
					})
					.finally(function(){
						currentPublish = undefined;
					});
				},
				function(err) {
					console.log(err);
					currentPublish = undefined;
					detailedReason = getDetailedErrorString('onPublishHTML:', {code: kTokenVerificationNetworkFailure}, err);
					sendPublishCompletionEvent(kFailure, {reason: kUnknownError, detailedReason: detailedReason});
				});
		}
		else
		{
			detailedReason = getDetailedErrorString('onPublishHTML:', {code: kUnextectedEventDataErr});
			sendPublishCompletionEvent(kFailure, {reason: kUnknownError, detailedReason: detailedReason});
		}
	} catch(err){
		detailedReason = getDetailedErrorString('onPublishHTML:', {code: kCatchOnPublishHTMLErr}, err);
		sendPublishCompletionEvent(kFailure, {reason: kUnknownError, detailedReason: detailedReason});
	}
}

function invokeHttpRequest(method, href, headers, body, options, sendProgress, donotLogRequest, numRetries, deferred) {
	deferred = deferred ? deferred : Q.defer();
	numRetries = typeof numRetries === 'undefined'? kMaxNumberOfRetriesForUpload: numRetries;
	var requestTime = new Date();
	httpService.invoke(method, href, headers, body, options, function(err, result) {
		var responseTime = new Date();
		var responseDuration = responseTime.getTime() - requestTime.getTime();
		if(err) {
			if (!donotLogRequest) {
				var httpRequestLogObject = {
					"time"				:	responseTime,
					"responseDuration"  	:   responseDuration,
					"method"				:	method,
					"href"				: 	href,
					"error"		        : 	(err.code ? err.code : ( err.message ? err.message :  err.toString()) )
				};
				publishOnlineLogObject.httpRequestLogData.push(httpRequestLogObject);
				//                 console.log(JSON.stringify(publishOnlineLogObject));
			}
			if (numRetries <= 0)
				deferred.reject(err);
			else
				invokeHttpRequest(method, href, headers, body, options, sendProgress, donotLogRequest, numRetries - 1, deferred);
		} else {
			if (!donotLogRequest) {
				var httpRequestLogObject = {
					"time"				:	Date(),
					"responseDuration"	:	responseTime.getTime() - requestTime.getTime(),
					"method"				:	method,
					"href"				: 	href,
					"statusCode"			: 	result.statusCode
				};
				publishOnlineLogObject.httpRequestLogData.push(httpRequestLogObject);
				//                 console.log(JSON.stringify(publishOnlineLogObject));
			}
			if (Math.floor(result.statusCode / 100) === 5 && numRetries > 0) // Retry in case of 5XX response and if we have retries left
				invokeHttpRequest(method, href, headers, body, options, sendProgress, donotLogRequest, numRetries - 1, deferred);
			else if (method === 'PUT' && result.statusCode === 409 && numRetries > 0) { // We are retrying a PUT request and previous PUT has created a partial file
				headers['If-match'] = '*';  //Force update the whole file
				invokeHttpRequest(method, href, headers, body, options, sendProgress, donotLogRequest, numRetries - 1, deferred);
			}
			else {
				if(sendProgress)
					deferred.notify(100);

				deferred.resolve(result);
			}
		}
	});
	return deferred.promise;
}

function PublishPackage(token) {
	httpService.setApiKey(api_key);
	httpService.setAuthToken(token);
	this._packageKey = utils.generateUuid();
	publishOnlineLogObject.resourcePath = this._packageKey;
	this._versionPrefix = 1;
	this._S3Requests = [];
}

PublishPackage.prototype.isRePublishing = function() {
	return !!(this._rePulishAssetID);
};

authCallback = function(httpSession) {
	console.log('Auth failure');
	cancelUpload();
	sendPublishCompletionEvent(kFailure, {reason: kAuthenticationError});
};

PublishPackage.prototype._generateNewPrefix = function(oldVersions) {
	var chars = '0123456789abcdefghijklmnopqrstuvwxyz';
	var result = '';
	for (var i = 4; i > 0; --i)
		result += chars[Math.round(Math.random() * (chars.length - 1))];
	// Check if it already been used
	var exist = false;
	for(var i = 0; i < oldVersions.length; i++) {
		if(oldVersions[i].prefix === result) {
			exist = true;
			break;
		}
	}
	if(exist) {
		return this._generateNewPrefix(oldVersions);
	}
	return result;
};

PublishPackage.prototype.setGoogleAnalyticsParams = function(measurementId, enableCookieConsentBannerOption, cookieConsentBannerText) {
    this._measurementId = measurementId;
	this._enableCookieConsentBannerOption = enableCookieConsentBannerOption;
	this._cookieConsentBannerText = cookieConsentBannerText;
}

PublishPackage.prototype.publish = function(packagePath, filesToUpload, title, description, machineTags, rePublishAssetID, idVersion, creatorName) {
	try{
		var start = new Date();
		console.log('Start Time: ' + start);
		this._totalTasks = 2; // Apart from file upload, there are 2 tasks of publishing to CP and finalize API.
		this._completedTasks = 0;
		this._title = title;
		this._description = description;
		this._totalPackageSize = 0;
		this._machineTags = machineTags;
		this._idVersion = idVersion;
		this._creatorName = creatorName;

		this._filesToUpload = [];
		this._rePulishAssetID = rePublishAssetID;

		publishOnlineLogObject.creatorName = creatorName;
		publishOnlineLogObject.idVersion = idVersion;
		publishOnlineLogObject.asset_id = rePublishAssetID;
		publishOnlineLogObject.isRepublishing   = this.isRePublishing();

		// On Mac, the temporary folder is created under /var/tmp. This folder is also present as a
		// symlink in /private/var/tmp. Sometime, we get folder with path /private/var/tmp/....
		// and files as /var/tmp/...
		// Hence, we resolve all files to its original path
		this._packagePath = getResolvedFilePath(packagePath);

		for(var i in filesToUpload) {
			var file = getResolvedFilePath(filesToUpload[i]);
			if(file.indexOf(this._packagePath) !== -1) {
				this._filesToUpload.push(file);
			} else {
				throwError('PublishPackage::publish:',  {code: kMismatchInFileAndPackagePathErr});
			}
		}

		var self = this;
		return this._findFileSizeLimit()
				.then(function(fileSizeLimit) {
					self._fileSizeLimit = fileSizeLimit;
					if(self.isRePublishing()) {
						return self._fetchRePublishMetadata()
								.then(function(metadata){
									if(!metadata.resource_path) {
										throw new Error('Older document. Can not be republished');
									}
									self._packageKey = metadata.resource_path;
									self._versions = (metadata.custom && metadata.custom.versions) ? metadata.custom.versions : [];
									self._versionPrefix = self._generateNewPrefix(self._versions);
									self._versions.push({
										prefix: self._versionPrefix
									});
								});
					}
				})
				.then(function() {
					var fileUploadPromises = [];
					for(var i in self._filesToUpload) {
						var filePath = self._filesToUpload[i];
						var relativePath = Path.relative(self._packagePath, filePath);
						if(Path.sep === '\\')	// Windows platform
							relativePath = relativePath.split(Path.sep).join('/');
						var fileKey = self._packageKey + '/' + self._versionPrefix + '/' + relativePath;
						console.log('Uploading file: ' + filePath + ' with key: ' + fileKey);
						var uploadPromise = self._uploadFile(filePath, fileKey);
						if(uploadPromise) {
							fileUploadPromises.push(uploadPromise);
						}
					}

					fileUploadPromises.push(self._uploadManifestFile());
					sendPublishPkgSize(self._totalPackageSize);
					return Q.all(fileUploadPromises);
				})
				.then(function() {
					if(self.isRePublishing()) {
						return self._rePublishToCp()
								.then(function(){
									return self._rePulishAssetID;
								});
					} else {
						return self._publishToCp()
								.then(function(result) {
									var code, errString;
									var end = new Date();
									console.log('Time taken: ' + (end.getTime() - start.getTime()) / 1000);
									if (!result || typeof result !== 'string') {
										code = {code: kInvalidResponseCPPublishErr, message: 'Response: ' + result};
										throwError('PublishPackage::publish:',  code);
									} else {
										var index = result.lastIndexOf('/');
										if (index !== -1) {
											asset_id = result.slice(index + 1);
											return asset_id;
										} else {
											code = {code: kMissingAssetIDErr, message: 'Response: ' + result};
											throwError('PublishPackage::publish:', code);
										}
									}
								});
					}
				})
				.then(function (asset_id) {
					return self._wrapProgressingTaskPromise(self._deleteOlderVersions(asset_id));
				})
				.catch(function(err){
					throwError('PublishPackage::publish:',  {code: kCatchCPPublishErr}, err);
				});
	} catch(err) {
		throwError('PublishPackage::publish: ', {code: kCatchPublishingPackageErr2}, err);
	}
};

PublishPackage.prototype._findFileSizeLimit = function() {
	return invokeHttpRequest('GET', hostname + GET_FILE_SIZE_URL, undefined, undefined, { responseType: 'json' }, false)
			.then(function(response) {
				if(response.statusCode != 200){
					var code = {code: kFindFileSizeLimitErr, message: 'StatusCode: ' + response.statusCode};
					throwError('PublishPackage::_findFileSizeLimit:', code);
				}
				try {
					var limit = JSON.parse(response.response);
					return limit.small_file_upload_limit;
				} catch (err) {
					throwError('PublishPackage::_findFileSizeLimit:', {code: kInvalidResponseFileSizeLimitErr}, err);
				}
			})
			.catch(function(err) {
				throwError('PublishPackage::_findFileSizeLimit:', {code: kCatchFindFileSizeLimitErr}, err);
			});
};

PublishPackage.prototype._fetchRePublishMetadata = function() {
	return invokeHttpRequest('GET', hostname + PUBLISH_CP_URL + this._rePulishAssetID, undefined, undefined, { responseType: 'json' }, false)
			.then(function(response){
				if(response.statusCode != 200){
					var code = {code: kFetchMetadataErr, message: 'StatusCode: ' + response.statusCode};
					throwError('PublishPackage::_fetchRePublishMetadata:', code);
				}
				try {
					var metadata = JSON.parse(response.response);
					return metadata;
				} catch (err) {
					throwError('PublishPackage::_fetchRePublishMetadata:', {code: kInvalidResponseMetadataErr}, err);
				}
			})
			.catch(function(err) {
				throwError('PublishPackage::_fetchRePublishMetadata:', {code: kCatchFetchMetadataErr}, err);
			});
};

PublishPackage.prototype._encodeURL = function(urlString) {
	if(urlString.indexOf("/") > -1){
		var fileName = urlString.substring(urlString.lastIndexOf("/")+1);
		var filePath = urlString.substring(0,urlString.lastIndexOf("/")+1);
		urlString = encodeURI(filePath) + encodeURIComponent(fileName);
	}
	return urlString;
}

PublishPackage.prototype._uploadFile = function(filepath, key) {
	var stats;
	try {
		stats = fs.statSync(filepath);
	} catch (err) {
		var code = {code: kFileReadErr, message: filepath};
		throwError('PublishPackage::_uploadFile:', code, err);
	}

	if (stats && stats.isFile()) {
		this._totalPackageSize = stats.size + this._totalPackageSize;
		if (stats.size < this._fileSizeLimit) {
			return this._uploadFileViaCP(filepath, key, stats.size);
		} else {
			return this._uploadFileDirectlyToS3(filepath, key, stats.size);
		}
	} else {
		throwError('PublishPackage::_uploadFile:', {code: kInvalidFilePathErr});
	}
};

PublishPackage.prototype._uploadFileViaCP = function(filepath, key, fileSize) {
	var file = new NetworkHttpService.FileContent(filepath);

	var contentType = getMimeByExt(filepath);
	var promise = invokeHttpRequest('PUT', hostname + UPLOAD_FILE_URL + this._encodeURL(key), {
		'content-type': contentType,
		'Content-Length': fileSize
	}, file, {
		responseType: 'json'
	}, true)
			.catch(function(err) {
				throwError('PublishPackage::_uploadFileViaCP', {code: kCatchUploadFileViaCPErr}, err);
			});
	var wrappedPromise = this._wrapProgressingTaskPromise(promise);
	this._totalTasks++;
	return wrappedPromise;
};

PublishPackage.prototype._uploadFileDirectlyToS3 = function(filepath, key, fileSize) {
	var uploadInfo = {};
	var self = this;
	var contentType = getMimeByExt(filepath);
	var s3CredPromise = invokeHttpRequest('POST', hostname + UPLOAD_FILE_URL + self._encodeURL(key) + UPLOAD_S3_URL_SUFFIX, {
		'content-type': contentType,
		'X-content-length': fileSize
	}, undefined, {
		responseType: 'text'
	}, false);

	// upload chunks to S3
	var newPromise = s3CredPromise.then(function(s3details) {
		try {
			s3details = JSON.parse(s3details.response);
		} catch (err) {
			throwError('PublishPackage::_uploadFileDirectlyToS3:', {code: kParsingS3DetailsFromCPErr}, err);
		}
		try{
			// Parse to get credentials for chunk upload
			uploadInfo.s3_access_details = s3details;
			uploadInfo.segments = [];
			var arn = s3details.asset_arn;
			arn = arn.slice(arn.lastIndexOf(':') + 1);
			var sep = arn.lastIndexOf('/');
			var s3bucket = arn.slice(0, sep);
			var s3key = arn.slice(sep + 1);
			var s3upload_id = s3details.upload_id;

			var S3 = new AWS.S3({
				accessKeyId: uploadInfo.s3_access_details.credentials.access_key,
				secretAccessKey: uploadInfo.s3_access_details.credentials.secret_key,
				sessionToken: uploadInfo.s3_access_details.credentials.session_token
			});

			var fd = fs.openSync(filepath, 'r');
			var partNum = 0;
			var promiseFunctions = [];
			var hash = crypt.createHash('md5');
			// Create promiseFunctions to upload chunks
			for (var rangeStart = 0; rangeStart < fileSize; rangeStart += S3_CHUNK_SIZE) {
				partNum++;
				var end = Math.min(rangeStart + S3_CHUNK_SIZE, fileSize);
				var size = end - rangeStart;
				var buf = new Buffer(size);
				var bytes = fs.readSync(fd, buf, 0, size, rangeStart);
				var partParams = {
					Body: buf.slice(0, bytes),
					Bucket: s3bucket,
					Key: s3key,
					PartNumber: String(partNum),
					UploadId: s3upload_id
				};

				hash.update(buf.slice(0, bytes));
				// Send a single part
				console.log('Uploading part: #', partParams.PartNumber, ', file: ', filepath);
				var chunkUploadPromiseFunc = 
				(function(S3, partParams, partNum){
					return function(){
						return self._UploadChunkToS3(S3, partParams, partNum)
						.then(function(result) {
							console.log('Success Uploading part: #', result.segment_number, ', file: ', filepath);
							uploadInfo.segments.push(result);
						});
					};
				})(S3, partParams, partNum);  // create a function which on execution will return an upload promise
				promiseFunctions.push(chunkUploadPromiseFunc);
			}

			fs.closeSync(fd);
			uploadInfo.md5 = hash.digest('base64');
			
			return Q.all(promiseFunctions.map(qlimit(function(func){ return func();})));
		} catch(err) {
			throwError('PublishPackage::_uploadFileDirectlyToS3:', {code: kReadingUploadingInChunkErr}, err);
		}
	})
		// call publish to CP, with all etags of uploaded chunks
			.then(
				function() {
					var reqBody = {
						s3_access_details: uploadInfo.s3_access_details,
						segments: uploadInfo.segments,
						md5: uploadInfo.md5,
						content_length: fileSize,
						type: contentType
					};
					return invokeHttpRequest('PUT', hostname + UPLOAD_FILE_URL + self._encodeURL(key) + UPLOAD_S3_URL_SUFFIX, {
						'content-type': 'application/json'
					}, JSON.stringify(reqBody), {
						responseType: 'json'
					}, true);
				},
				function(err) {
					throwError('PublishPackage::_uploadFileDirectlyToS3:', {code: kUploadChunkS3Err}, err);
				}
			)
		// parse the result to get Asset-ID
			.then(function(result) {
				if (result.statusCode !== 201 && result.statusCode != 200) {
					var code = {code: kFinalizingUploadS3WithStatusCodeErr, message: 'StatusCode: ' + result.statusCode};
					throwError('PublishPackage::_uploadFileDirectlyToS3:', code);
				}
				console.log('Success in finalizing upload to S3');
				return result;
			}, function(err) {
				throwError('PublishPackage::_uploadFileDirectlyToS3:', {code: kFinalizingUploadS3Err}, err);
			})
			.catch(function(err) {
				throwError('PublishPackage::_uploadFileDirectlyToS3', {code: kCatchUploadFileDirectlyToS3Err}, err);
			});

	var wrappedPromise = this._wrapProgressingTaskPromise(newPromise);
	this._totalTasks += Math.ceil(fileSize/S3_CHUNK_SIZE + 1);
	return wrappedPromise;
};

PublishPackage.prototype._createMd5 = function(filepath) {
	var deferred = Q.defer();
	var hash = crypt.createHash('md5');
	var stream = fs.createReadStream(filepath);
	stream.on('data', function (data) {
		hash.update(data);
	});

	stream.on('end', function () {
		var digest = hash.digest('base64');
		deferred.resolve(digest);
	});

	stream.on('error', function(err) {
		deferred.reject(err);
	});

	return deferred.promise;
};

PublishPackage.prototype._UploadChunkToS3 = function(s3, reqParams, chunkNum) {
	var self = this;
	var req = s3.uploadPart(reqParams);
	var deferred = Q.defer();
	req.send(function(err, result) {
		if (err || !result.ETag) {
			deferred.reject(err || new Error('PublishPackage:_UploadChunkToS3 Inavlid response in uploading chunk to S3'));
		} else {
			// Return the result
			var pos = self._S3Requests.indexOf(req);
			if (pos >= 0) {
				self._S3Requests.splice(pos, 1);
			} else {
				console.log('S3 request is not found in the list of pending requests');
			}
			deferred.resolve({
				segment_number: chunkNum,
				entity_tag: result.ETag
			});
			self._onTaskCompletion();
		}
	});
	this._S3Requests.push(req);
	return deferred.promise;
};

PublishPackage.prototype._uploadManifestFile = function() {
	var manifestFilePath = Path.join(this._packagePath, MANIFEST_FILE);
	if(!fs.existsSync(manifestFilePath)) {
		throwError('PublishPackage::_uploadManifestFile', {code: kManifestFileNotExistsErr}, new Error(""));
	}

	var manifestFileKey = this._packageKey + '/' + MANIFEST_FILE;
	var file = new NetworkHttpService.FileContent(manifestFilePath);

	var contentType = getMimeByExt(manifestFilePath);
	var headers = {
		'content-type': contentType
	};

	if(this.isRePublishing()) {
		headers['If-match'] = '*';
	}

	var promise = invokeHttpRequest('PUT', hostname + UPLOAD_FILE_URL + manifestFileKey, headers, file, {
		responseType: 'json'
	}, true)
			.catch(function(err) {
				throwError('PublishPackage::_uploadManifestFile', {code: kManifestFileNotExistsErr}, err);
			});
	var wrappedPromise = this._wrapProgressingTaskPromise(promise);
	this._totalTasks++;
	return wrappedPromise;
};

PublishPackage.prototype._rePublishArtwork = function() {
	var artworkPath = this._packageKey + '/' + this._versionPrefix + coverImagePath;
	console.log('Artwork path key: ' + artworkPath);

	var body = {
		resource_path: artworkPath
	};
	var self = this;
	var updateArtworkUrl = hostname + PUBLISH_CP_URL + this._rePulishAssetID + '/artwork';
	return invokeHttpRequest('PUT', updateArtworkUrl, {
		'content-type': 'application/json',
		'community_id': COMMUNITY_ID
	}, JSON.stringify(body), {
		responseType: 'json'
	}, true)
			.then(function(response) {
				if (!response ||
						response.statusCode !== 200) {
					var code = {code: kRePublishArtworkResponseErr, message: 'StatusCode: ' + response.statusCode};
					throwError('PublishPackage::_rePublishArtwork:', code);
				}
				return response;
			}, function(err) {
				throwError('PublishPackage::_rePublishArtwork', {code: kRePublishArtworkCatchErr}, err);
			});
};

PublishPackage.prototype._rePublishToCp = function() {
	var artworkPath = this._packageKey + '/' + this._versionPrefix + coverImagePath;
	console.log('Artwork path key: ' + artworkPath);

	var self = this;
	return self._rePublishArtwork()
			.then(function() {
				var body = {
					description: self._description,
					title: self._title,
					machine_tags: self._machineTags,
					artwork: {
						resource_path: artworkPath
					},
					custom: {
						version_prefix: self._versionPrefix,
						versions: self._versions,
						idVersion: self._idVersion,
						measurementId: self._measurementId,
						enableCookieConsentBannerOption: self._enableCookieConsentBannerOption,
						cookieConsentBannerText: self._cookieConsentBannerText
					}
				};

				var updateMetadataUrl = hostname + PUBLISH_CP_URL + self._rePulishAssetID;
				var updateMetadataPromise = invokeHttpRequest('PUT', updateMetadataUrl, {
					'content-type': 'application/json',
					'community_id': COMMUNITY_ID
				}, JSON.stringify(body), {
					responseType: 'json'
				}, true);

				return self._wrapProgressingTaskPromise(updateMetadataPromise);
			})
			.then(function(response) {
				if (!response || response.statusCode !== 200) {
					var code = {code: kPublishedWithNoProperResponseErr, message: 'StatusCode: ' + response.statusCode};
					throwError('PublishPackage::_publishToCp:', code);
				}
				return response;
			}, function(err) {
				throwError('PublishPackage::_rePublishToCp', {code: kRePublishMetadataCatchErr}, err);
			});
};

PublishPackage.prototype._publishToCp = function() {
	var urlPath = PUBLISH_CP_URL;
	var artworkPath = this._packageKey + '/' + this._versionPrefix + coverImagePath;
	console.log('Artwork path key: ' + artworkPath);
	var body = {
		metadata: {
			category_id: "other",
			creator_ids: [],
			description: this._description,
			sub_category_ids: [],
			tags: [],
			machine_tags: this._machineTags,
			title: this._title,
			artwork: {
				resource_path: artworkPath
			},
			undiscoverable: false,
			custom: {
				version_prefix: this._versionPrefix,
				idVersion: this._idVersion,
				versions: [{
					prefix: this._versionPrefix
				}],
				measurementId: this._measurementId,
				enableCookieConsentBannerOption: this._enableCookieConsentBannerOption,
				cookieConsentBannerText: this._cookieConsentBannerText
			}
		},
		resource_path: this._packageKey,
		resource_type: PUBLISH_RESOURCE_TYPE
	};
	var self = this;
	var publishPromise = invokeHttpRequest('POST', hostname + urlPath, {
		'content-type': 'application/json',
		'community_id': COMMUNITY_ID
	}, JSON.stringify(body), {
		responseType: 'json'
	}, true)
			.catch(function(err){
				throwError('PublishPackage::_publishToCp', {code: kPublishTOCPCatchErr}, err);
			});
	return this._wrapProgressingTaskPromise(publishPromise)
			.then(function(response) {
				if (response &&
						response.statusCode === 201 &&
						response.headers &&
						response.headers.location) {
					console.log('Published to CP');
					// Send progress info
					return (response.headers.location);
				} else {
					var code = {code: kPublishedWithNoProperResponseErr, message: 'StatusCode: ' + response.statusCode};
					throwError('PublishPackage::_publishToCp:', code);
				}
			});
};

PublishPackage.prototype._wrapProgressingTaskPromise = function(promise){
	var self = this;
	var filepath = promise.filepath;
	return promise.then(function(result){
		return result;
	}, function(err){
		throwError('PublishPackage::_wrapProgressingTaskPromise:', {code: kTaskFailedErr}, err);
	}, function(progressInfo){
		self._onTaskCompletion(progressInfo);
	});
};

PublishPackage.prototype._onTaskCompletion = function(progressInfo) {
	this._completedTasks++;
	progressInfo = ((this._completedTasks / this._totalTasks) * 100);
	console.log('Progress info: ' + progressInfo);
	sendProgressEvent(progressInfo);
};

PublishPackage.prototype._onTaskError = function(progressInfo) {
	console.log('Aborting the publish');
	this.abort();
};

PublishPackage.prototype.abort = function() {
	httpService.abortAll();
	for(var i in this._S3Requests){
		this._S3Requests[i].abort();
	}
	this._S3Requests = [];
};

PublishPackage.prototype._deleteOlderVersions = function (asset_id) {
	var finalizeHeaders = {
		"docId": asset_id,
		"versionPrefix": this._versionPrefix
	};
	return invokeHttpRequest('POST', webDocHost + FINALIZE_ENDPOINT, finalizeHeaders, undefined, { responseType: 'json' }, true)
			.then(function () {
				return asset_id;
			})
			.catch(function (err) {
				console.log("Error while trying to call finalize API: " + err);
				return asset_id;
			});
};
